/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

Components.utils.import("resource://calendar/modules/calUtils.jsm");
Components.utils.import("resource://calendar/modules/calItipUtils.jsm");

/**
 * This bar lives inside the message window.
 * Its lifetime is the lifetime of the main thunderbird message window.
 */
var ltnImipBar = {

    actionFunc: null,
    itipItem: null,
    foundItems: null,

    /**
     * Thunderbird Message listener interface, hide the bar before we begin
     */
    onStartHeaders: function onImipStartHeaders() {
      ltnImipBar.resetBar();
    },

    /**
     * Thunderbird Message listener interface
     */
    onEndHeaders: function onImipEndHeaders() {

    },

    /**
     * Load Handler called to initialize the imip bar
     * NOTE: This function is called without a valid this-context!
     */
    load: function ltnImipOnLoad() {
        // Add a listener to gMessageListeners defined in msgHdrViewOverlay.js
        gMessageListeners.push(ltnImipBar);

        // We need to extend the HideMessageHeaderPane function to also hide the
        // message header pane. Otherwise, the imip bar will still be shown when
        // changing folders.
        ltnImipBar.tbHideMessageHeaderPane = HideMessageHeaderPane;
        HideMessageHeaderPane = function ltnHideMessageHeaderPane() {
            ltnImipBar.resetBar();
            ltnImipBar.tbHideMessageHeaderPane.apply(null, arguments);
        };

        // Set up our observers
        Services.obs.addObserver(ltnImipBar, "onItipItemCreation", false);
    },

    /**
     * Unload handler to clean up after the imip bar
     * NOTE: This function is called without a valid this-context!
     */
    unload: function ltnImipOnUnload() {
        removeEventListener("messagepane-loaded", ltnImipBar.load, true);
        removeEventListener("messagepane-unloaded", ltnImipBar.unload, true);

        ltnImipBar.resetBar();
        Services.obs.removeObserver(ltnImipBar, "onItipItemCreation");
    },

    observe: function ltnImipBar_observe(subject, topic, state) {
        if (topic == "onItipItemCreation") {
            let itipItem = null;
            try {
                if (!subject) {
                    let sinkProps = msgWindow.msgHeaderSink.properties;
                    // This property was set by lightningTextCalendarConverter.js
                    itipItem = sinkProps.getPropertyAsInterface("itipItem", Components.interfaces.calIItipItem);
                }
            } catch (e) {
                // This will throw on every message viewed that doesn't have the
                // itipItem property set on it. So we eat the errors and move on.

                // XXX TODO: Only swallow the errors we need to. Throw all others.
            }
            if (!itipItem || !gMessageDisplay.displayedMessage) {
                return;
            }

            let imipMethod = gMessageDisplay.displayedMessage.getStringProperty("imip_method");
            cal.itip.initItemFromMsgData(itipItem, imipMethod, gMessageDisplay.displayedMessage);

            let imipBar = document.getElementById("imip-bar");
            imipBar.setAttribute("collapsed", "false");
            imipBar.setAttribute("label",  cal.itip.getMethodText(itipItem.receivedMethod));

            cal.itip.processItipItem(itipItem, ltnImipBar.setupOptions);
        }
    },

    /**
     * Hide the imip bar and reset the itip item.
     */
    resetBar: function ltnResetImipBar() {
        document.getElementById("imip-bar").collapsed = true;
        ltnImipBar.resetButtons();

        // Clear our iMIP/iTIP stuff so it doesn't contain stale information.
        cal.itip.cleanupItipItem(ltnImipBar.itipItem);
        ltnImipBar.itipItem = null;
    },

    /**
     * Resets all buttons and its menuitems, all buttons are hidden thereafter
     */
    resetButtons: function ltnResetImipButtons() {
        let buttons = ltnImipBar.getButtons();
        buttons.forEach(hideElement);
        buttons.forEach(function(aButton) ltnImipBar.getMenuItems(aButton).forEach(showElement));
    },

    /**
     * Provides a list of all available buttons
     */
    getButtons: function ltnGetButtons() {
        let buttons = [];
        let nl = document.getElementById("imip-view-toolbar")
                         .getElementsByTagName("toolbarbutton");
        if (nl != null && nl.length > 0) {
            for (let button of nl) {
                buttons.push(button);
            }
        }
        return buttons;
    },

    /**
     * Provides a list of available menuitems of a button
     *
     * @param aButton        button node
     */
    getMenuItems: function ltnGetMenuItems(aButton) {
        let items = [];
        let mitems = aButton.getElementsByTagName("menuitem");
        if (mitems != null && mitems.length > 0) {
            for (let mitem of mitems) {
                items.push(mitem);
            }
        }
        return items;
    },

    /**
     * Checks and converts button types based on available menuitems of the buttons
     * to avoid dropdowns which are empty or only replicating the default button action
     * Should be called once the buttons are set up
     */
    conformButtonType: function ltnConformButtonType() {
        // check only needed on visible and not simple buttons
        let buttons = ltnImipBar.getButtons()
                                .filter(function(aElement) aElement.hasAttribute("type") && !aElement.hidden);
        // change button if appropriate
        for (let button of buttons) {
            let items = ltnImipBar.getMenuItems(button).filter(function(aItem) !aItem.hidden);
            if (button.type == "menu" && items.length == 0) {
                // hide non functional buttons
                button.hidden = true;
            } else if (button.type == "menu-button") {
                if (items.length == 0 ||
                    (items.length == 1 &&
                     button.hasAttribute("oncommand") &&
                     items[0].hasAttribute("oncommand") &&
                     button.getAttribute("oncommand")
                           .endsWith(items[0].getAttribute("oncommand")))) {
                   // convert to simple button
                   button.removeAttribute("type");
                }
            }
        }
    },

    /**
     * This is our callback function that is called each time the itip bar UI needs updating.
     * NOTE: This function is called without a valid this-context!
     *
     * @param itipItem      The iTIP item to set up for
     * @param rc            The status code from processing
     * @param actionFunc    The action function called for execution
     * @param foundItems    An array of items found while searching for the item
     *                        in subscribed calendars
     */
    setupOptions: function setupOptions(itipItem, rc, actionFunc, foundItems) {
        let imipBar =  document.getElementById("imip-bar");
        let data = cal.itip.getOptionsText(itipItem, rc, actionFunc, foundItems);

        if (Components.isSuccessCode(rc)) {
            ltnImipBar.itipItem = itipItem;
            ltnImipBar.actionFunc = actionFunc;
            ltnImipBar.foundItems = foundItems;
        }

        imipBar.setAttribute("label", data.label);

        // menu items are visible by default, let's hide what's not available
        data.hideMenuItems.forEach(function(aElementId) hideElement(document.getElementById(aElementId)));
        // buttons are hidden by default, let's make required buttons visible
        data.buttons.forEach(function(aElementId) showElement(document.getElementById(aElementId)));
        // adjust button style if necessary
        ltnImipBar.conformButtonType();
    },

    executeAction: function ltnExecAction(partStat, extendResponse) {
        if (partStat == null) {
            partStat = '';
        }
        if (partStat == "X-SHOWDETAILS") {
            let items = ltnImipBar.foundItems;
            if (items && items.length) {
                let item = items[0].isMutable ? items[0] : items[0].clone();
                modifyEventWithDialog(item);
            }
        } else {
            if (extendResponse) {
                // Open an extended response dialog to enable the user to add a comment, make a
                // counter proposal, delegate the event or interact in another way.
                // Instead of a dialog, this might be implemented as a separate container inside the
                // imip-overlay as proposed in bug 458578
                //
                // If implemented as a dialog, the OL compatibility decision should be incorporated
                // therein too and the itipItems's autoResponse set to auto subsequently
                // to prevent a second popup during imip transport processing.
            }
            let delmgr = Components.classes["@mozilla.org/calendar/deleted-items-manager;1"]
                                   .getService(Components.interfaces.calIDeletedItems);
            let items = ltnImipBar.itipItem.getItemList({});
            if (items && items.length) {
                let delTime = delmgr.getDeletedDate(items[0].id);
                let dialogText = ltnGetString("lightning", "confirmProcessInvitation");
                let dialogTitle = ltnGetString("lightning", "confirmProcessInvitationTitle");
                if (delTime && !Services.prompt.confirm(window, dialogTitle, dialogText)) {
                    return false;
                }
            }

            if (cal.itip.promptCalendar(ltnImipBar.actionFunc.method, ltnImipBar.itipItem, window)) {
                // hide the buttons now, to disable pressing them twice...
                ltnImipBar.resetButtons();

                let opListener = {
                    onOperationComplete: function ltnItipActionListener_onOperationComplete(aCalendar,
                                                                                            aStatus,
                                                                                            aOperationType,
                                                                                            aId,
                                                                                            aDetail) {
                        // For now, we just state the status for the user something very simple
                        let imipBar = document.getElementById("imip-bar");
                        let label = cal.itip.getCompleteText(aStatus, aOperationType);
                        imipBar.setAttribute("label", label);

                        if (!Components.isSuccessCode(aStatus)) {
                            showError(label);
                        }
                    },
                    onGetResult: function ltnItipActionListener_onGetResult(aCalendar,
                                                                            aStatus,
                                                                            aItemType,
                                                                            aDetail,
                                                                            aCount,
                                                                            aItems) {
                    }
                };

                try {
                    ltnImipBar.actionFunc(opListener, partStat);
                } catch (exc) {
                    Components.utils.reportError(exc);
                }
                return true;
            }
        }
        return false;
    }
};

addEventListener("messagepane-loaded", ltnImipBar.load, true);
addEventListener("messagepane-unloaded", ltnImipBar.unload, true);
